<?php
/**
 * UTECH SMS Gateway - Receive SMS API Endpoint
 * Uukow Data Exchange System
 * 
 * This endpoint receives SMS data from the Flutter app,
 * parses payment information, and triggers data bundle provisioning.
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

require_once 'config.php';
require_once 'db_connect.php';
require_once 'functions.php';

// Handle OPTIONS request for CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Method not allowed. Use POST.',
    ]);
    exit;
}

// Get POST data
$input = file_get_contents('php://input');
$data = json_decode($input, true);

// Log incoming request
logMessage("Received SMS data: " . $input);

// Validate JSON
if (json_last_error() !== JSON_ERROR_NONE) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Invalid JSON data',
    ]);
    exit;
}

// Check if this is a test request
if (isset($data['test']) && $data['test'] === true) {
    logMessage("Test connection request received");
    echo json_encode([
        'success' => true,
        'message' => 'Connection successful',
        'timestamp' => date('Y-m-d H:i:s'),
    ]);
    exit;
}

// Validate required fields
$required_fields = ['sender', 'message', 'timestamp', 'token'];
foreach ($required_fields as $field) {
    if (!isset($data[$field]) || empty($data[$field])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => "Missing required field: $field",
        ]);
        exit;
    }
}

// Verify secret token
if ($data['token'] !== SECRET_TOKEN) {
    logMessage("Invalid token received: " . $data['token']);
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'message' => 'Invalid authentication token',
    ]);
    exit;
}

// Extract data
$sender = trim($data['sender']);
$message = trim($data['message']);
$timestamp = $data['timestamp'];

// Parse SMS message
$parsed = parseSmsMessage($message, $sender);

logMessage("Parsed SMS - Amount: {$parsed['amount']}, Sender: {$parsed['sender_number']}, Transaction: {$parsed['transaction_id']}, Operator: {$parsed['operator']}");

// Check if this is a payment SMS
if (!$parsed['is_payment']) {
    logMessage("Not a payment SMS, ignoring");
    echo json_encode([
        'success' => true,
        'message' => 'SMS received but not a payment message',
        'action' => 'ignored',
    ]);
    exit;
}

// Validate parsed data
if (empty($parsed['amount']) || empty($parsed['sender_number'])) {
    logMessage("Failed to parse payment details");
    echo json_encode([
        'success' => false,
        'message' => 'Could not parse payment details from SMS',
    ]);
    exit;
}

// Check for duplicate transaction
if (transactionExists($conn, $parsed['transaction_id'])) {
    logMessage("Duplicate transaction detected: " . $parsed['transaction_id']);
    echo json_encode([
        'success' => true,
        'message' => 'Transaction already processed',
        'action' => 'duplicate',
    ]);
    exit;
}

// Save SMS to database
$sms_id = saveSms($conn, $sender, $message, $timestamp, $parsed);

if (!$sms_id) {
    logMessage("Failed to save SMS to database");
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Failed to save SMS to database',
    ]);
    exit;
}

logMessage("SMS saved with ID: $sms_id");

// Provision data bundle
$provision_result = provisionDataBundle(
    $conn,
    $parsed['operator'],
    $parsed['sender_number'],
    $parsed['amount'],
    $sms_id
);

// Prepare response
$response = [
    'success' => true,
    'message' => 'SMS processed successfully',
    'sms_id' => $sms_id,
    'parsed_data' => [
        'amount' => $parsed['amount'],
        'sender_number' => $parsed['sender_number'],
        'transaction_id' => $parsed['transaction_id'],
        'operator' => $parsed['operator'],
    ],
    'provisioning' => $provision_result,
];

logMessage("Response: " . json_encode($response));

echo json_encode($response);

// Close database connection
$conn->close();

