-- =====================================================
-- UTECH SMS Gateway Database Tables
-- For Uukow Data Exchange System
-- =====================================================

USE uukow_data_exchange;

-- =====================================================
-- Table: received_sms
-- Stores all SMS messages received from the mobile app
-- =====================================================
CREATE TABLE IF NOT EXISTS `received_sms` (
  `id` INT(11) NOT NULL AUTO_INCREMENT,
  `sender` VARCHAR(100) NOT NULL COMMENT 'SMS sender identifier (e.g., Hormuud, Somnet)',
  `message` TEXT NOT NULL COMMENT 'Full SMS message content',
  `timestamp` VARCHAR(50) NOT NULL COMMENT 'Timestamp when SMS was received',
  `amount` DECIMAL(10,2) DEFAULT NULL COMMENT 'Parsed payment amount',
  `sender_number` VARCHAR(20) DEFAULT NULL COMMENT 'Phone number of the payment sender',
  `transaction_id` VARCHAR(100) DEFAULT NULL COMMENT 'Transaction/Reference ID',
  `operator` VARCHAR(50) DEFAULT NULL COMMENT 'Operator name (hormuud, somnet, somtel)',
  `processed` TINYINT(1) DEFAULT 0 COMMENT '1 if processed, 0 if pending',
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  INDEX `idx_transaction_id` (`transaction_id`),
  INDEX `idx_sender_number` (`sender_number`),
  INDEX `idx_processed` (`processed`),
  INDEX `idx_created_at` (`created_at`),
  INDEX `idx_operator` (`operator`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Stores all received SMS messages';

-- =====================================================
-- Table: provision_logs
-- Stores all data bundle provisioning actions
-- =====================================================
CREATE TABLE IF NOT EXISTS `provision_logs` (
  `id` INT(11) NOT NULL AUTO_INCREMENT,
  `sms_id` INT(11) DEFAULT NULL COMMENT 'Reference to received_sms.id',
  `phone_number` VARCHAR(20) NOT NULL COMMENT 'Phone number that received the bundle',
  `operator` VARCHAR(50) NOT NULL COMMENT 'Operator name (hormuud, somnet, somtel)',
  `bundle_size` VARCHAR(20) NOT NULL COMMENT 'Data bundle size (e.g., 1GB, 2GB)',
  `amount` DECIMAL(10,2) NOT NULL COMMENT 'Payment amount',
  `method` VARCHAR(20) DEFAULT NULL COMMENT 'Provisioning method (API or SMS)',
  `status` VARCHAR(20) DEFAULT NULL COMMENT 'Provisioning status (success, failed)',
  `response` TEXT DEFAULT NULL COMMENT 'API/SMS gateway response',
  `retry_count` INT(11) DEFAULT 0 COMMENT 'Number of retry attempts',
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  INDEX `idx_sms_id` (`sms_id`),
  INDEX `idx_phone_number` (`phone_number`),
  INDEX `idx_status` (`status`),
  INDEX `idx_created_at` (`created_at`),
  INDEX `idx_operator` (`operator`),
  FOREIGN KEY (`sms_id`) REFERENCES `received_sms`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Stores data bundle provisioning logs';

-- =====================================================
-- Sample Data (for testing)
-- =====================================================

-- Example 1: Hormuud payment
-- INSERT INTO `received_sms` 
-- (`sender`, `message`, `timestamp`, `amount`, `sender_number`, `transaction_id`, `operator`, `processed`) 
-- VALUES 
-- ('Hormuud', 'You have received $1.00 from 615123456. Ref: TXN12345', '2025-01-15 10:30:00', 1.00, '615123456', 'TXN12345', 'hormuud', 1);

-- Example 2: Somnet payment
-- INSERT INTO `received_sms` 
-- (`sender`, `message`, `timestamp`, `amount`, `sender_number`, `transaction_id`, `operator`, `processed`) 
-- VALUES 
-- ('Somnet', 'Payment of $2.00 from 618987654. Transaction: ABC98765', '2025-01-15 11:00:00', 2.00, '618987654', 'ABC98765', 'somnet', 1);

-- =====================================================
-- Useful Queries for Monitoring
-- =====================================================

-- View all unprocessed SMS
-- SELECT * FROM received_sms WHERE processed = 0 ORDER BY created_at DESC;

-- View recent provisioning logs
-- SELECT * FROM provision_logs ORDER BY created_at DESC LIMIT 50;

-- Count successful vs failed provisions
-- SELECT status, COUNT(*) as count FROM provision_logs GROUP BY status;

-- Daily statistics
-- SELECT 
--   DATE(created_at) as date,
--   COUNT(*) as total_sms,
--   SUM(CASE WHEN processed = 1 THEN 1 ELSE 0 END) as processed,
--   SUM(amount) as total_amount
-- FROM received_sms
-- GROUP BY DATE(created_at)
-- ORDER BY date DESC;

-- View provisioning by operator
-- SELECT 
--   operator,
--   COUNT(*) as total,
--   SUM(CASE WHEN status = 'success' THEN 1 ELSE 0 END) as successful,
--   SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed
-- FROM provision_logs
-- GROUP BY operator;

-- =====================================================
-- Maintenance Queries
-- =====================================================

-- Clean up old logs (older than 90 days)
-- DELETE FROM provision_logs WHERE created_at < DATE_SUB(NOW(), INTERVAL 90 DAY);
-- DELETE FROM received_sms WHERE created_at < DATE_SUB(NOW(), INTERVAL 90 DAY);

-- Reprocess failed provisions
-- UPDATE received_sms SET processed = 0 
-- WHERE id IN (
--   SELECT sms_id FROM provision_logs WHERE status = 'failed'
-- );

