<?php
/**
 * UTECH SMS Gateway - Admin Dashboard
 * View received SMS messages and provisioning logs
 */

require_once 'config.php';
require_once 'db_connect.php';

// Get statistics
$stats_sql = "
    SELECT 
        (SELECT COUNT(*) FROM received_sms) as total_sms,
        (SELECT COUNT(*) FROM received_sms WHERE processed = 1) as processed_sms,
        (SELECT COUNT(*) FROM provision_logs) as total_provisions,
        (SELECT COUNT(*) FROM provision_logs WHERE status = 'success') as successful_provisions,
        (SELECT SUM(amount) FROM received_sms WHERE processed = 1) as total_amount,
        (SELECT COUNT(*) FROM received_sms WHERE DATE(created_at) = CURDATE()) as today_sms
";
$stats = fetchOne($conn, $stats_sql);

// Get recent SMS
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$per_page = 20;
$offset = ($page - 1) * $per_page;

$recent_sms_sql = "SELECT * FROM received_sms ORDER BY created_at DESC LIMIT ? OFFSET ?";
$recent_sms = fetchAll($conn, $recent_sms_sql, [$per_page, $offset], 'ii');

$total_sms = $stats['total_sms'];
$total_pages = ceil($total_sms / $per_page);

// Get recent provisions
$recent_provisions_sql = "
    SELECT p.*, r.message 
    FROM provision_logs p 
    LEFT JOIN received_sms r ON p.sms_id = r.id 
    ORDER BY p.created_at DESC 
    LIMIT 20
";
$recent_provisions = fetchAll($conn, $recent_provisions_sql);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>UTECH SMS Gateway - Admin Dashboard</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f5f5;
            color: #212121;
        }
        
        .header {
            background: linear-gradient(135deg, #1976D2 0%, #0D47A1 100%);
            color: white;
            padding: 20px 0;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 0 20px;
        }
        
        h1 {
            font-size: 28px;
            margin-bottom: 5px;
        }
        
        .subtitle {
            opacity: 0.9;
            font-size: 14px;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin: 30px 0;
        }
        
        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .stat-card h3 {
            font-size: 14px;
            color: #757575;
            margin-bottom: 10px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .stat-card .value {
            font-size: 32px;
            font-weight: bold;
            color: #1976D2;
        }
        
        .stat-card.success .value {
            color: #4CAF50;
        }
        
        .stat-card.warning .value {
            color: #FFC107;
        }
        
        .card {
            background: white;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 30px;
            overflow: hidden;
        }
        
        .card-header {
            padding: 20px;
            background: #f9f9f9;
            border-bottom: 1px solid #e0e0e0;
        }
        
        .card-header h2 {
            font-size: 20px;
            color: #212121;
        }
        
        .card-body {
            padding: 0;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        th {
            background: #f5f5f5;
            padding: 15px;
            text-align: left;
            font-weight: 600;
            color: #424242;
            font-size: 13px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        td {
            padding: 15px;
            border-bottom: 1px solid #f0f0f0;
            font-size: 14px;
        }
        
        tr:hover {
            background: #fafafa;
        }
        
        .badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
        }
        
        .badge.success {
            background: #E8F5E9;
            color: #2E7D32;
        }
        
        .badge.failed {
            background: #FFEBEE;
            color: #C62828;
        }
        
        .badge.pending {
            background: #FFF3E0;
            color: #EF6C00;
        }
        
        .badge.api {
            background: #E3F2FD;
            color: #1565C0;
        }
        
        .badge.sms {
            background: #F3E5F5;
            color: #6A1B9A;
        }
        
        .pagination {
            display: flex;
            justify-content: center;
            gap: 10px;
            padding: 20px;
        }
        
        .pagination a {
            padding: 8px 16px;
            background: #1976D2;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            transition: background 0.3s;
        }
        
        .pagination a:hover {
            background: #0D47A1;
        }
        
        .pagination a.active {
            background: #0D47A1;
        }
        
        .pagination a.disabled {
            background: #ccc;
            pointer-events: none;
        }
        
        .operator {
            font-weight: 600;
            text-transform: capitalize;
        }
        
        .operator.hormuud {
            color: #D32F2F;
        }
        
        .operator.somnet {
            color: #1976D2;
        }
        
        .operator.somtel {
            color: #388E3C;
        }
        
        .refresh-btn {
            float: right;
            padding: 8px 16px;
            background: #1976D2;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            font-size: 14px;
            transition: background 0.3s;
        }
        
        .refresh-btn:hover {
            background: #0D47A1;
        }
        
        .message-preview {
            max-width: 400px;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        @media (max-width: 768px) {
            .stats-grid {
                grid-template-columns: 1fr;
            }
            
            table {
                font-size: 12px;
            }
            
            th, td {
                padding: 10px 8px;
            }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="container">
            <h1>🚀 UTECH SMS Gateway</h1>
            <div class="subtitle">Uukow Data Exchange - Admin Dashboard</div>
        </div>
    </div>
    
    <div class="container">
        <!-- Statistics Cards -->
        <div class="stats-grid">
            <div class="stat-card">
                <h3>Total SMS</h3>
                <div class="value"><?php echo number_format($stats['total_sms']); ?></div>
            </div>
            
            <div class="stat-card success">
                <h3>Processed</h3>
                <div class="value"><?php echo number_format($stats['processed_sms']); ?></div>
            </div>
            
            <div class="stat-card">
                <h3>Provisions</h3>
                <div class="value"><?php echo number_format($stats['total_provisions']); ?></div>
            </div>
            
            <div class="stat-card success">
                <h3>Successful</h3>
                <div class="value"><?php echo number_format($stats['successful_provisions']); ?></div>
            </div>
            
            <div class="stat-card">
                <h3>Total Amount</h3>
                <div class="value">$<?php echo number_format($stats['total_amount'], 2); ?></div>
            </div>
            
            <div class="stat-card warning">
                <h3>Today's SMS</h3>
                <div class="value"><?php echo number_format($stats['today_sms']); ?></div>
            </div>
        </div>
        
        <!-- Recent SMS Messages -->
        <div class="card">
            <div class="card-header">
                <h2>Recent SMS Messages</h2>
                <a href="?refresh=1" class="refresh-btn">🔄 Refresh</a>
            </div>
            <div class="card-body">
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Sender</th>
                            <th>Message</th>
                            <th>Amount</th>
                            <th>Phone</th>
                            <th>Transaction ID</th>
                            <th>Operator</th>
                            <th>Status</th>
                            <th>Received</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($recent_sms as $sms): ?>
                        <tr>
                            <td><?php echo $sms['id']; ?></td>
                            <td><?php echo htmlspecialchars($sms['sender']); ?></td>
                            <td class="message-preview" title="<?php echo htmlspecialchars($sms['message']); ?>">
                                <?php echo htmlspecialchars($sms['message']); ?>
                            </td>
                            <td><?php echo $sms['amount'] ? '$' . number_format($sms['amount'], 2) : '-'; ?></td>
                            <td><?php echo htmlspecialchars($sms['sender_number'] ?: '-'); ?></td>
                            <td><?php echo htmlspecialchars($sms['transaction_id'] ?: '-'); ?></td>
                            <td>
                                <?php if ($sms['operator']): ?>
                                    <span class="operator <?php echo $sms['operator']; ?>">
                                        <?php echo htmlspecialchars($sms['operator']); ?>
                                    </span>
                                <?php else: ?>
                                    -
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if ($sms['processed']): ?>
                                    <span class="badge success">Processed</span>
                                <?php else: ?>
                                    <span class="badge pending">Pending</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo date('M d, Y H:i', strtotime($sms['created_at'])); ?></td>
                        </tr>
                        <?php endforeach; ?>
                        <?php if (empty($recent_sms)): ?>
                        <tr>
                            <td colspan="9" style="text-align: center; padding: 40px; color: #999;">
                                No SMS messages received yet
                            </td>
                        </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
                
                <?php if ($total_pages > 1): ?>
                <div class="pagination">
                    <?php if ($page > 1): ?>
                        <a href="?page=<?php echo $page - 1; ?>">← Previous</a>
                    <?php else: ?>
                        <a href="#" class="disabled">← Previous</a>
                    <?php endif; ?>
                    
                    <a href="#" class="active">Page <?php echo $page; ?> of <?php echo $total_pages; ?></a>
                    
                    <?php if ($page < $total_pages): ?>
                        <a href="?page=<?php echo $page + 1; ?>">Next →</a>
                    <?php else: ?>
                        <a href="#" class="disabled">Next →</a>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Recent Provisioning Logs -->
        <div class="card">
            <div class="card-header">
                <h2>Recent Provisioning Logs</h2>
            </div>
            <div class="card-body">
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Phone Number</th>
                            <th>Operator</th>
                            <th>Bundle</th>
                            <th>Amount</th>
                            <th>Method</th>
                            <th>Status</th>
                            <th>Date</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($recent_provisions as $provision): ?>
                        <tr>
                            <td><?php echo $provision['id']; ?></td>
                            <td><?php echo htmlspecialchars($provision['phone_number']); ?></td>
                            <td>
                                <span class="operator <?php echo $provision['operator']; ?>">
                                    <?php echo htmlspecialchars($provision['operator']); ?>
                                </span>
                            </td>
                            <td><strong><?php echo htmlspecialchars($provision['bundle_size']); ?></strong></td>
                            <td>$<?php echo number_format($provision['amount'], 2); ?></td>
                            <td>
                                <span class="badge <?php echo strtolower($provision['method']); ?>">
                                    <?php echo htmlspecialchars($provision['method']); ?>
                                </span>
                            </td>
                            <td>
                                <span class="badge <?php echo $provision['status']; ?>">
                                    <?php echo htmlspecialchars($provision['status']); ?>
                                </span>
                            </td>
                            <td><?php echo date('M d, Y H:i', strtotime($provision['created_at'])); ?></td>
                        </tr>
                        <?php endforeach; ?>
                        <?php if (empty($recent_provisions)): ?>
                        <tr>
                            <td colspan="8" style="text-align: center; padding: 40px; color: #999;">
                                No provisioning logs yet
                            </td>
                        </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</body>
</html>
<?php $conn->close(); ?>

